// index.js or app.js

const express = require("express");
const dotenv = require("dotenv");
const cors = require("cors");
const morgan = require("morgan");
const jwt = require("jsonwebtoken");

const { connectToDatabase } = require("./utils/connectDb");
const { loginHandler } = require("./controllers/login");
const { authenticateToken } = require("./middlewares/authentication");

// Load environment variables from .env file
dotenv.config();

// Create Express app
const app = express();
const port = process.env.PORT || 8080;
app.use(express.json());

// Enable Cross-Origin Resource Sharing (CORS)
app.use(cors());

// Use Morgan for HTTP request logging
app.use(morgan("dev")); // 'dev' is a predefined format string

app.post("/login", loginHandler);

function getTablesSet(choice) {
  // Replace these with your actual table names
  const tables = {
    doctors: {
      main: "webapp_doctors",
      relation: "webapp_docrelation",
      number: "webapp_docnumber",
      brancherel: "webapp_docbrancherel",
      branche: "webapp_docbranche",
    },
    hospitals: {
      main: "webapp_hospitals",
      relation: "webapp_hosrelation",
      number: "webapp_hosnumber",
      brancherel: "webapp_hosbrancherel",
      branche: "webapp_hosbranche",
    },
  };
  return tables[choice];
}

const special_data = {
  1: "الجراحة الــعامـة",
  2: "جراحة الاوعية الدموية",
  3: "جراحة الاطفـال",
  4: "جراحة القلب والصدر",
  5: "جراحة التجميل والحـــروق",
  6: "العظام",
  7: "جراحة المسالك البولية والكـلي",
  8: "الباطنة العامـة",
  9: "باطنة كلي",
  10: "باطنة قلب وأوعية دموية",
  11: "باطنة حميات وجهاز هضمي وكبد",
  12: "الامراض الصدريــة",
  13: "أنف وأذن وحنجرة",
  14: "عيون / رمد",
  15: "جلدية وأمراض تناسلية",
  16: "روماتيزم ومناعة",
  17: "غدد صماء",
  18: "الأورام والطب النووي",
  19: "علاج الالام",
  20: "جراحة المخ والأعصاب",
  21: "مخ واعصاب وطب نفسي",
  22: "النساء والتوليد",
  23: "الأطفال",
  24: "تغذية علاجية",
  25: "طب وجراحة الأسنان",
  26: "العلاج الطبيعي",
};
async function addInfo(type, body, connection) {
  const tables = getTablesSet(type);
  let includeSpecialFields = type === "doctors";

  // Prepare data for insertion
  let name = body.name || null;
  let special = includeSpecialFields ? body.special || null : null;
  let notes = body.note || null;
  let specialName =
    includeSpecialFields && special ? special_data[special] || "" : "";

  let insertMainQuery;
  let insertData;

  if (includeSpecialFields) {
    insertMainQuery = `INSERT INTO ${tables.main} (name, special, specialname, note) VALUES (?, ?, ?, ?)`;
    insertData = [name, special, specialName, notes];
  } else {
    insertMainQuery = `INSERT INTO ${tables.main} (name, note) VALUES (?, ?)`;
    insertData = [name, notes];
  }

  // Insert main record
  const [mainTableResult] = await connection.execute(
    insertMainQuery,
    insertData
  );
  let mainTableId = mainTableResult.insertId;

  // Handling extra_info data
  const extraInfoData = body.extra_info && body.extra_info.data;
  if (extraInfoData) {
    for (const data of extraInfoData) {
      const { branch_name, country_value, numbers } = data;

      // Insert branch
      let insertBranch = `INSERT INTO ${tables.branche} (branche) VALUES (?)`;
      const [branchResult] = await connection.execute(insertBranch, [
        branch_name,
      ]);
      let branchId = branchResult.insertId;

      // Insert relation
      let insertRelation = `INSERT INTO ${tables.relation} (relation_name_id, relation_branche_id, relation_country_id) VALUES (?, ?, ?)`;
      await connection.execute(insertRelation, [
        mainTableId,
        branchId,
        country_value,
      ]);

      // Insert numbers
      for (const number of numbers) {
        let insertNumber = `INSERT INTO ${tables.number} (number) VALUES (?)`;
        const [numberResult] = await connection.execute(insertNumber, [
          number.value,
        ]);
        let numberId = numberResult.insertId;

        // Insert branch-number relation
        let insertDocBranchRel = `INSERT INTO ${tables.brancherel} (relation_number_id, relation_branche_id) VALUES (?, ?)`;
        await connection.execute(insertDocBranchRel, [numberId, branchId]);
      }
    }
  }
}

async function postHandler(req, res) {
  try {
    await addInfo(req, res);
    res.status(200).send("record has been added successfully");
  } catch (error) {
    console.log("error", error);
    res.status(500).send(error.message);
  }
}
async function getDoctorDetails(connection, choice, id) {
  const doctorQuery = id
    ? `SELECT * FROM ${choice.main} WHERE id = ?`
    : `SELECT * FROM ${choice.main}`;

  const [doctors] = await connection.query(doctorQuery, id ? [id] : []);
  let response = [];

  for (let doctor of doctors) {
    let data = {
      name: doctor.name,
      id: doctor.id,
      note: doctor.note,
      special: doctor.special, // Assuming this field exists in both doctors and hospitals tables
      extra_info: {
        data: [],
      },
    };

    const relationQuery = `
          SELECT dr.*, b.branche as branch_name, c.name as country_name 
          FROM ${choice.relation} dr
          JOIN ${choice.branche} b ON dr.relation_branche_id = b.id
          LEFT JOIN sql_store.countries c ON dr.relation_country_id = c.id
          WHERE dr.relation_name_id = ?`;

    const [docrelations] = await connection.query(relationQuery, [doctor.id]);

    for (let docrelation of docrelations) {
      const numberQuery = `
              SELECT dn.id, dn.number 
              FROM ${choice.brancherel} dbr
              JOIN ${choice.number} dn ON dbr.relation_number_id = dn.id
              WHERE dbr.relation_branche_id = ?`;

      const [numbers] = await connection.query(numberQuery, [
        docrelation.relation_branche_id,
      ]);

      let mappedNumbers = numbers.map((number) => ({
        id: number.id,
        value: number.number,
      }));

      let branchData = {
        branch_name: docrelation.branch_name,
        branch_value: docrelation.relation_branche_id,
        country_name: docrelation.country_name,
        country_value: docrelation.relation_country_id,
        numbers: mappedNumbers,
      };

      data.extra_info.data.push(branchData);
    }

    response.push(data);
  }

  return response;
}

app.get("/data/:type?/:id?", authenticateToken, async (req, res) => {
  const id = req.params.id || null;
  const type = req.params.type || null;
  const connection = await connectToDatabase();
  try {
    const tableSet = getTablesSet(type);
    if (id) {
      // Check if the doctor exists
      const [existing] = await connection.query(
        `SELECT * FROM ${tableSet.main} WHERE id = ?`,
        [id]
      );
      if (existing.length === 0) {
        return res.status(404).json({ message: "Doctor not found" });
      }
    }
    const doctors = await getDoctorDetails(connection, tableSet, id);
    res.json(doctors);
  } catch (error) {
    console.log("error", error);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    connection.end();
  }
});

// Modify postHandler to use the new addInfo
app.post("/data/:type", authenticateToken, async (req, res) => {
  try {
    const connection = await connectToDatabase();
    await addInfo(req.params.type, req.body, connection);
    connection.end();
    res.status(200).send("Record has been added successfully");
  } catch (error) {
    console.log("error", error);
    res.status(500).send(error.message);
  }
});
const updateData = async (req, res) => {
  const { type, id } = req.params;
  const connection = await connectToDatabase();
  const tables = getTablesSet(type);
  const body = req.body;

  try {
    await connection.beginTransaction();

    if (type == "doctors") {
      // Update the main record
      let updateMain = `UPDATE ${tables.main} SET name = ?, note = ?, special = ? WHERE id = ?`;
      await connection.query(updateMain, [
        body.name,
        body.note,
        body.special,
        id,
      ]);
    } else {
      // Update the main record
      let updateMain = `UPDATE ${tables.main} SET name = ?, note = ? WHERE id = ?`;
      await connection.query(updateMain, [body.name, body.note, id]);
    }

    // Fetch existing branch IDs
    const [oldBranches] = await connection.query(
      `SELECT relation_branche_id FROM ${tables.relation} WHERE relation_name_id = ?`,
      [id]
    );
    const branchIds = oldBranches.map((branch) => branch.relation_branche_id);

    // Delete the relations first
    await connection.query(
      `DELETE FROM ${tables.relation} WHERE relation_name_id = ?`,
      [id]
    );

    if (branchIds.length > 0) {
      // Delete branch-number relations
      await connection.query(
        `DELETE FROM ${tables.brancherel} WHERE relation_branche_id IN (?)`,
        [branchIds]
      );

      // Delete numbers
      await connection.query(
        `DELETE FROM ${tables.number} WHERE id IN (SELECT relation_number_id FROM ${tables.brancherel} WHERE relation_branche_id IN (?))`,
        [branchIds]
      );

      // Finally, delete the branches
      await connection.query(`DELETE FROM ${tables.branche} WHERE id IN (?)`, [
        branchIds,
      ]);
    }

    // Add new branches
    for (const data of body.extra_info.data) {
      const { branch_name, country_value, numbers } = data;

      // Insert new branch
      let insertBranch = `INSERT INTO ${tables.branche} (branche) VALUES (?)`;
      const [branchResult] = await connection.execute(insertBranch, [
        branch_name,
      ]);
      let branchId = branchResult.insertId;

      // Insert relation
      let insertRelation = `INSERT INTO ${tables.relation} (relation_name_id, relation_branche_id, relation_country_id) VALUES (?, ?, ?)`;
      await connection.execute(insertRelation, [id, branchId, country_value]);

      // Insert numbers
      for (const number of numbers) {
        let insertNumber = `INSERT INTO ${tables.number} (number) VALUES (?)`;
        const [numberResult] = await connection.execute(insertNumber, [
          number.value,
        ]);
        let numberId = numberResult.insertId;

        // Insert branch-number relation
        let insertDocBranchRel = `INSERT INTO ${tables.brancherel} (relation_number_id, relation_branche_id) VALUES (?, ?)`;
        await connection.execute(insertDocBranchRel, [numberId, branchId]);
      }
    }

    await connection.commit();
    res.json({
      message: `${
        type.charAt(0).toUpperCase() + type.slice(1)
      } updated successfully`,
    });
  } catch (error) {
    await connection.rollback();
    console.error("Error during update operation:", error);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    connection.end();
  }
};

// Define your Express route for updating data
app.put("/data/:type/:id", authenticateToken, updateData);

// DELETE handler for deleting a doctor and its related data
app.delete("/data/:type/:id", authenticateToken, async (req, res) => {
  const { type, id } = req.params;
  const connection = await connectToDatabase();
  const tables = getTablesSet(type);

  try {
    // Start a transaction to ensure atomicity
    await connection.beginTransaction();

    // Fetch existing branch IDs
    const [oldBranches] = await connection.query(
      `SELECT relation_branche_id FROM ${tables.relation} WHERE relation_name_id = ?`,
      [id]
    );
    const branchIds = oldBranches.map((branch) => branch.relation_branche_id);

    // Delete the relations first
    await connection.query(
      `DELETE FROM ${tables.relation} WHERE relation_name_id = ?`,
      [id]
    );

    if (branchIds.length > 0) {
      // Delete branch-number relations
      await connection.query(
        `DELETE FROM ${tables.brancherel} WHERE relation_branche_id IN (?)`,
        [branchIds]
      );

      // Delete numbers
      await connection.query(
        `DELETE FROM ${tables.number} WHERE id IN (SELECT relation_number_id FROM ${tables.brancherel} WHERE relation_branche_id IN (?))`,
        [branchIds]
      );

      // Finally, delete the branches
      await connection.query(`DELETE FROM ${tables.branche} WHERE id IN (?)`, [
        branchIds,
      ]);
    }
    // Attempt to delete the doctor and check if the doctor was found and deleted
    const [deleteDoctorResult] = await connection.query(
      `DELETE FROM ${tables.main} WHERE id = ?`,
      [id]
    );
    if (deleteDoctorResult.affectedRows === 0) {
      // If no rows were affected, the doctor was not found
      await connection.rollback(); // Undo any changes
      return res.status(404).json({ message: "Doctor not found" });
    }

    // Commit the transaction
    await connection.commit();

    res.json({
      message: "Doctor and all related data have been deleted successfully",
    });
  } catch (error) {
    // In case of error, rollback the transaction
    await connection.rollback();
    console.error("Error during the delete operation:", error);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    // End the database connection
    connection.end();
  }
});

// Start the server
app.listen(port, () => {
  console.log(`Server running at http://localhost:${port}`);
});
